/**
 * \file: mspin_demo_iap2_connection.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN iAP2 Connection
 *
 * \component: MSPIN
 *
 * \author: Bui Le Thuan / thuan.buile@vn.bosch.com
 *          Thilo Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#ifndef MSPIN_IAP2_SUPPORT_DISABLED

#include <mspin_demo_iap2_connection.h>
#include <mspin_demo_iap2_types.h>
#include <mspin_demo_iap2_configuration_pfcfg.h>
#include <mspin_logging.h>
#include <pthread.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <glob.h>
#include <sys/prctl.h>


mspin_demo_iap2_iOSDevice_t Iap2Connection::m_iAP2Device;
iOSDeviceReady Iap2Connection::m_Connect_CB;
iOSDeviceDisconnected Iap2Connection::m_Disconnected_CB;
EAPSessionStarted Iap2Connection::m_EAPSessionStarted_CB;
EAPSessionStopped Iap2Connection::m_EAPSessionStopped_CB;
BOOL Iap2Connection::m_EndMonitoring = FALSE;

U16 mspin_demo_USBHostModeMsgSentByAcc[] = {                    \
        IAP2_MSG_ID_START_POWER_UPDATES,                        \
        IAP2_MSG_ID_STOP_POWER_UPDATES,                         \
        IAP2_MSG_ID_REQUEST_APP_LAUNCH,                         \
        IAP2_MSG_ID_POWER_SOURCE_UPDATE                         \
};

U16 mspin_demo_USBHostModeMsgRecvFromDevice[] = {               \
        IAP2_MSG_ID_POWER_UPDATE                               \
};

Iap2Connection::Iap2Connection()
{
    m_Connect_CB = NULL;
    m_Disconnected_CB = NULL;
    m_EAPSessionStarted_CB = NULL;
    m_EAPSessionStopped_CB = NULL;

    m_MyspinMQFD = -1;
    m_PollThreadID = 0;
    m_AppThreadID = 0;
}

Iap2Connection::~Iap2Connection()
{

}

S32 Iap2Connection::updateDeviceState_CB(iAP2Device_t* iap2Device, iAP2DeviceState_t state, void* context)
{
    S32 rc = IAP2_OK;
    (void)context;

    switch (state)
    {
        case iAP2NotConnected:
        {
            m_iAP2Device.testDeviceState = iAP2NotConnected;
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(device=%p) new state = iAP2NotConnected",
                    __FUNCTION__, iap2Device);

            if (m_Disconnected_CB)
            {
                m_Disconnected_CB(context);
            }

            break;
        }
        case iAP2LinkConnected:
        {
            m_iAP2Device.testDeviceState = iAP2LinkConnected;
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) new state = iAP2LinkConnected",
                    __FUNCTION__, iap2Device);
            break;
        }
        case iAP2AuthenticationPassed:
        {
            m_iAP2Device.testDeviceState = iAP2AuthenticationPassed;
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) new state = iAP2AuthenticationPassed",
                    __FUNCTION__, iap2Device);
            break;
        }
        case iAP2IdentificationPassed:
        {
            m_iAP2Device.testDeviceState = iAP2IdentificationPassed;
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) new state = iAP2IdentificationPassed",
                    __FUNCTION__, iap2Device);
            break;
        }
        case iAP2DeviceReady:
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) new state = iAP2DeviceReady",
                    __FUNCTION__, iap2Device);
            m_iAP2Device.testDeviceState = iAP2DeviceReady;
            if (m_Connect_CB)
            {
                m_Connect_CB(context, TRUE);
            }
            break;
        }
        case iAP2LinkiAP1DeviceDetected:
        {
            m_iAP2Device.testDeviceState = iAP2LinkiAP1DeviceDetected;
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) new state = iAP2LinkiAP1DeviceDetected",
                    __FUNCTION__, iap2Device);
            break;
        }
        case iAP2TransportConnected:
        {
            m_iAP2Device.testDeviceState = iAP2TransportConnected;
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s() new device=%p state = iAP2TransportConnected",
                    __FUNCTION__, iap2Device);
            break;
        }
        default:
        {
            m_iAP2Device.testDeviceState = iAP2ComError;
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s(device=%p) new state = unknown (%d)",
                    __FUNCTION__, iap2Device, state);
            rc = IAP2_CTL_ERROR;
            break;
        }
    }

    return rc;
}

S32 Iap2Connection::startEANativeTransport_CB(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context)
{
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(device=%p, iOSAppId=%d, ctx=%p)",
            __FUNCTION__, iap2Device, iAP2iOSAppIdentifier, context);

    mspin_demo_connectionParameter_t *pConnectionParam = (mspin_demo_connectionParameter_t*)context;

    if (m_EAPSessionStarted_CB && pConnectionParam)
    {
        pConnectionParam->iOSAppIdentifier = iAP2iOSAppIdentifier;
        pConnectionParam->sinkEndpoint = sinkEndpoint;
        pConnectionParam->sourceEndpoint = sourceEndpoint;

        m_EAPSessionStarted_CB(context);
    }

    return IAP2_OK;
}

S32 Iap2Connection::stopEANativeTransport_CB(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context)
{
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(device=%p, iOSAppId=%d, sinkEp?%d, sourcEp=%d, ctx=%p)",
            __FUNCTION__, iap2Device, iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint, context);

    if (m_EAPSessionStopped_CB)
    {
        m_EAPSessionStopped_CB(context);
    }

    return IAP2_OK;
}

S32 Iap2Connection::authenticationFailed_CB(iAP2Device_t* iap2Device, iAP2AuthenticationFailedParameter* authParameter, void* context)
{
    (void)authParameter;
    (void)iap2Device;

    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(...,ctx=%p)", __FUNCTION__, context);

    return IAP2_CTL_ERROR;
}

S32 Iap2Connection::authenticationSucceeded_CB(iAP2Device_t* iap2Device, iAP2AuthenticationSucceededParameter* authParameter, void* context)
{
    (void)authParameter;
    (void)iap2Device;

    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(..., ctx=%p)", __FUNCTION__, context);

    return IAP2_OK;
}

S32 Iap2Connection::identificationAccepted_CB(iAP2Device_t* iap2Device, iAP2IdentificationAcceptedParameter* idParameter, void* context)
{
    (void)idParameter;
    (void)context;
    (void)iap2Device;

    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(..., ctx=%p)", __FUNCTION__, context);

    return IAP2_OK;
}

S32 Iap2Connection::identificationRejected_CB(iAP2Device_t* iap2Device, iAP2IdentificationRejectedParameter* idParameter, void* context)
{
    (void)iap2Device;

    mspin_log_printLn(eMspinVerbosityError,
            "%s(..., ctx=%p) ERROR", __FUNCTION__, context);

    Iap2Utilites::parseIdRejectedMsg(idParameter);

    return IAP2_CTL_ERROR;
}

S32 Iap2Connection::powerUpdate_CB(iAP2Device_t* iap2Device, iAP2PowerUpdateParameter* powerupdateParameter, void* context)
{
    S32 rc = IAP2_CTL_ERROR;

    (void)iap2Device;
    (void)powerupdateParameter;

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(..., ctx=%p)", __FUNCTION__, context);

    if (powerupdateParameter->iAP2MaximumCurrentDrawnFromAccessory_count != 0)
    {
        rc = IAP2_OK;
    }
    if (powerupdateParameter->iAP2AccessoryPowerMode_count != 0)
    {
        rc = IAP2_OK;
    }
    if (powerupdateParameter->iAP2DeviceBatteryWillChargeIfPowerIsPresent_count != 0)
    {
        rc = IAP2_OK;
    }

    return rc;
}


//Init Gadget Configuration
S32 Iap2Connection::initGadgetConfiguration(iAP2_usbg_config_t* usb_gadget_configuration, iAP2AccessoryInfo_t* p_iAP2AccessoryInfo)
{
    char c_srate[] = {"44100,48000"};
    char p_srate[] = {"44100,48000"};

    S32 rc = IAP2_OK;
    U32 i;
    /* support only one mySPIN instance */
    U8 iAP2GadgetName[STRING_MAX - 1]           = {MYSPIN_DEMO_IAP2_GADGET_NAME};
    U8 iAP2FFS_InstanceName[STRING_MAX - 5]     = {MYSPIN_DEMO_IAP2_FUNCTION_FS_NAME};

    usb_gadget_configuration->iAP2GadgetName                = (U8*)strndup((const char*)iAP2GadgetName, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2FFS_InstanceName          = (U8*)strndup((const char*)iAP2FFS_InstanceName, STRING_MAX - 4);
    usb_gadget_configuration->iAP2AccessoryName             = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryName, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessoryModelIdentifier  = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryModelIdentifier, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessoryManufacturer     = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryManufacturer, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessorySerialNumber     = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessorySerialNumber, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessoryVendorId         = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryVendorId, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessoryProductId        = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryProductId, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2AccessoryBcdDevice        = (U8*)strndup((const char*)p_iAP2AccessoryInfo->iAP2AccessoryBcdDevice, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2ConfigFS_MountLocation    = (U8*)strndup((const char*)MYSPIN_DEMO_IAP2_CONFIG_FS_PATH, (STRING_MAX - 1) );
    usb_gadget_configuration->iAP2UdcDeviceName             = (U8*)strndup((const char*)m_UdcParam.pUdcDevice, (STRING_MAX - 1) );

    if( (usb_gadget_configuration->iAP2GadgetName == NULL)                  ||
        (usb_gadget_configuration->iAP2FFS_InstanceName == NULL)            ||
        (usb_gadget_configuration->iAP2AccessoryName == NULL)               ||
        (usb_gadget_configuration->iAP2AccessoryModelIdentifier == NULL)    ||
        (usb_gadget_configuration->iAP2AccessoryManufacturer == NULL)       ||
        (usb_gadget_configuration->iAP2AccessorySerialNumber == NULL)       ||
        (usb_gadget_configuration->iAP2AccessoryVendorId == NULL)           ||
        (usb_gadget_configuration->iAP2AccessoryProductId == NULL)          ||
        (usb_gadget_configuration->iAP2AccessoryBcdDevice == NULL)          ||
        (usb_gadget_configuration->iAP2ConfigFS_MountLocation == NULL)      ||
        (usb_gadget_configuration->iAP2UdcDeviceName == NULL) )
    {
        rc = IAP2_ERR_NO_MEM;
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not Enough Memory", __FUNCTION__);
    }
    if(rc == IAP2_OK)
    {
        U8 iAP2UAC2_InstanceName[STRING_MAX - 6]  = {MYSPIN_DEMO_IAP2_UAC2GADGET_NAME};

        usb_gadget_configuration->iAP2UAC2_InstanceName = (U8*)strndup((const char*)iAP2UAC2_InstanceName, STRING_MAX - 5);
        usb_gadget_configuration->iAP2_UAC2_Attrs = (usbg_f_uac2_attrs*)calloc(1, sizeof(usbg_f_uac2_attrs) );
        if( (usb_gadget_configuration->iAP2_UAC2_Attrs == NULL) ||
          (usb_gadget_configuration->iAP2UAC2_InstanceName == NULL) )
        {
          rc = IAP2_ERR_NO_MEM;
          mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not Enough Memory", __FUNCTION__);
        }
        else
        {
          usb_gadget_configuration->iAP2_UAC2_Attrs->c_chmask     = 3;
          usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate_def  = 44100;
          usb_gadget_configuration->iAP2_UAC2_Attrs->c_ssize      = 2;
          usb_gadget_configuration->iAP2_UAC2_Attrs->delay_tout   = 80;
          usb_gadget_configuration->iAP2_UAC2_Attrs->p_chmask     = 0;
          usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate_def  = 44100;
          usb_gadget_configuration->iAP2_UAC2_Attrs->p_ssize      = 2;
          usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate      = strdup(c_srate);
          usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate      = strdup(p_srate);
        }
    }
    if(rc == IAP2_OK)
    {
        usb_gadget_configuration->iAP2FFSConfig.iOSAppNames = (U8**) calloc(p_iAP2AccessoryInfo->iAP2SupportediOSAppCount, sizeof(U8*));
        usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier = (U8*) calloc(p_iAP2AccessoryInfo->iAP2SupportediOSAppCount, sizeof(U8));
        if( (usb_gadget_configuration->iAP2FFSConfig.iOSAppNames == NULL) ||
          (usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier == NULL) )
        {
            rc = IAP2_ERR_NO_MEM;
        }
    }
    if( (rc == IAP2_OK) )
    {
        usb_gadget_configuration->iAP2FFSConfig.nativeTransport = TRUE;
        usb_gadget_configuration->iAP2FFSConfig.iOSAppCnt = p_iAP2AccessoryInfo->iAP2SupportediOSAppCount;
        for(i = 0; ( (i < p_iAP2AccessoryInfo->iAP2SupportediOSAppCount) && (rc == IAP2_OK) ); i++)
        {
            usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[i] = (U8*)strdup((const char*)p_iAP2AccessoryInfo->iAP2iOSAppInfo[i].iAP2iOSAppName);
            if(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[i] == NULL)
            {
                rc = IAP2_ERR_NO_MEM;
            }
            else
            {
                usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier[i] = p_iAP2AccessoryInfo->iAP2iOSAppInfo[i].iAP2iOSAppIdentifier;
            }
        }
    }

    return rc;
}

void Iap2Connection::iap2FreePtr(void** iap2PtrToFree)
{
    if(*iap2PtrToFree != NULL)
    {
        free(*iap2PtrToFree);
        *iap2PtrToFree = NULL;
    }
}

void Iap2Connection::deInitGadgetConfiguration(iAP2_usbg_config_t* usb_gadget_configuration)
{
    U32 i;

    iap2FreePtr((void**)&usb_gadget_configuration->iAP2GadgetName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2FFS_InstanceName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryModelIdentifier);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryManufacturer);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessorySerialNumber);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryVendorId);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryProductId);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2AccessoryBcdDevice);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2ConfigFS_MountLocation);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2UdcDeviceName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2UAC2_InstanceName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2_UAC2_Attrs);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2NCM_InstanceName);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2FFSConfig.initEndPoint);
    iap2FreePtr((void**)&usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier);
    if(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames != NULL)
    {
        for(i = 0; i < usb_gadget_configuration->iAP2FFSConfig.iOSAppCnt; i++)
        {
            iap2FreePtr((void**)&usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[i]);
        }
        iap2FreePtr((void**)&usb_gadget_configuration->iAP2FFSConfig.iOSAppNames);
    }
}

//Switch USB OTG port to gadget
S32 Iap2Connection::switchOTGPortToDevice(void)
{
    iAP2USBRoleSwitchStatus usb_status = IAP2_USB_ROLE_SWITCH_OK;

    S32 rc;
    int res;
    char *mount_fs_param;

    memset(&m_UdcParam, 0, sizeof(udcParamInfo_t));

    memset(&m_RoleSwitchInfo, 0, sizeof(iAP2USBRoleSwitchInfo));

    rc = IAP2_OK;

    /* --- switch to host mode --- */

    /* fill info and store udev path belonging to info */

    /* depreciated.
     * use vbusPower. */
    m_RoleSwitchInfo.powerGPIO = -1;

    if (rc == IAP2_OK)
    {
        /* FIXME check if the device is the only device connected to the port
         * if not switching the otg mode might cause problems for the other device
         */

        /* FIXME when supporting multiple otg ports we need to find the correct vbus
         * by searching the parent usb port of the device
         */
        rc = iap2FindVbus(&m_RoleSwitchInfo);
        mspin_log_printLn(eMspinVerbosityInfo,
             "%s() iap2FindVbus  = %d | %s", __FUNCTION__, rc, m_RoleSwitchInfo.vbusPower);
        /* if we are not on im6-qi, we might don't have vbus */
        if (rc != IAP2_OK) {
            mspin_log_printLn(eMspinVerbosityInfo,
                 "%s() No vbus available", __FUNCTION__);
            /* continue until we know that we not directly connected to USB OTG port */
            rc = IAP2_OK;
        }

        /* vendorId and productId of an Apple device */
        //Vendor and product ID of an Apple device
        m_RoleSwitchInfo.vendorId = MYSPIN_DEMO_APPL_APPLE_VENDOR_ID;
        m_RoleSwitchInfo.productId = MYSPIN_DEMO_APPL_APPLE_PRODUCT_ID_MIN;
        m_RoleSwitchInfo.serialNumber = (char*)m_iAP2InitParameter.iAP2DeviceId;
        m_RoleSwitchInfo.mode = IAP2_USB_ROLE_SWITCH_WITHOUT_DIGITAL_IPOD_OUT; //no iOS in the car

        m_RoleSwitchInfo.otgGlob = iap2GetSwtichOtgGlobPath();
    }

    if (rc == IAP2_OK)
    {
    	usb_status = static_cast<iAP2USBRoleSwitchStatus>(iap2SwitchToHostMode(&m_RoleSwitchInfo, &m_UdcParam));
        mspin_log_printLn(eMspinVerbosityInfo,
             "%s() iap2SwitchToHostMode = %d", __FUNCTION__, usb_status);
    }
    if (IAP2_USB_ROLE_SWITCH_OK == usb_status)
    {
        if (iap2IsKernel314()  || iap2IsKernel4xx())
        {
            /* kernel module libcomposite depends on module configfs */
            rc = m_gadgetLoader.LoadKernelModule(NULL, CONFIGFS_MODULE_NAME, strlen(CONFIGFS_MODULE_NAME));
        }
        if(IAP2_OK == rc)
        {
            if(IAP2_OK != m_gadgetLoader.LoadKernelModule(NULL, LIBCOMPOSITE_MODULE_NAME, strlen(LIBCOMPOSITE_MODULE_NAME)))
            {
                rc = IAP2_CTL_ERROR;
            }
        }
        mspin_log_printLn(eMspinVerbosityInfo,
             "%s() load modules  = %d", __FUNCTION__, rc);
    }
    else if (IAP2_USB_ROLE_SWITCH_VENDOR_REQUEST_NOT_SUPPORTED == usb_status)
	{
		rc = IAP2_ERR_USB_ROLE_SWITCH_UNSUP;
	}
	else
	{
		rc = IAP2_ERR_USB_ROLE_SWITCH_FAILED;
	}

    if(IAP2_OK == rc)
    {
        mount_fs_param = NULL;

        res = mount(CONFIG_FS_NAME, CONFIG_FS_PATH, CONFIG_FS_TYPE, MS_NOEXEC, mount_fs_param);
        if (0 != res){
            rc = IAP2_CTL_ERROR;
        }

        /* after mount we need to set the mode again */
        res = chmod(CONFIG_FS_PATH, 0750);
        if (0 != res){
            rc = IAP2_CTL_ERROR;
        }

        rc = initGadgetConfiguration(&m_iAP2UsbgConfig, m_iAP2InitParameter.p_iAP2AccessoryInfo);

        if (IAP2_OK == rc)
        {
           rc = iAP2InitializeGadget(&m_iAP2UsbgConfig);
           mspin_log_printLn(eMspinVerbosityInfo, "%s() iAP2InitializeGadget returned with rc=%d", __FUNCTION__, rc);
        }

        if (IAP2_OK == rc)
        {
            res = mkdir(FUNCTION_FS_PATH, 0750);
            if (0 != res)
            {
                rc = IAP2_CTL_ERROR;
            }

            res = mount(FUNCTION_FS_NAME, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, MS_NOEXEC, mount_fs_param);
            if (0 != res)
            {
                rc = IAP2_CTL_ERROR;
            }

            res = chmod(FUNCTION_FS_PATH, 0750);
            if (0 != res)
            {
                rc = IAP2_CTL_ERROR;
            }
        }

        if (IAP2_OK == rc)
        {
            /* After mount set the mode again */
            res = chmod(MYSPIN_DEMO_IAP2_ACC_INFO_INIT_ENDPOINT, 0777);
            if (0!= res)
            {
                rc = IAP2_CTL_ERROR;
            }
        }

        if (IAP2_OK == rc)
        {
            U8 initEndPoint[] = {MYSPIN_DEMO_IAP2_ACC_INFO_INIT_ENDPOINT};

            m_iAP2UsbgConfig.iAP2FFSConfig.initEndPoint = (U8*)strdup((const char*)initEndPoint);
            rc = iAP2InitializeFFSGadget(&m_iAP2UsbgConfig);
            mspin_log_printLn(eMspinVerbosityInfo, "%s() iAP2InitializeFFSGadget returned with rc=%d", __FUNCTION__, rc);
        }
    }
    return rc;
}

//Switch USB OTG port to host
S32 Iap2Connection::switchOTGPortToHost(void)
{
    S32 rc = IAP2_OK;
    /* FIXME for supporting multiple ones we need to have a list of infos
     * and search for correct one
     */

    rc = iAP2DeInitializeGadget();
    mspin_log_printLn(eMspinVerbosityInfo, "%s() iAP2DeInitializeGadget returned with rc=%d", __FUNCTION__, rc);

    deInitGadgetConfiguration(&m_iAP2UsbgConfig);

	/* unmount and unload Kernel modules */
    umount(CONFIG_FS_PATH);
    rmdir(CONFIG_FS_PATH);
    umount(FUNCTION_FS_PATH);
    rmdir(FUNCTION_FS_PATH);

    if (iap2IsKernel314() || iap2IsKernel4xx()) {
        m_gadgetLoader.UnloadKernelModule(USB_F_FS_MODULE_NAME);
        m_gadgetLoader.UnloadKernelModule(USB_F_UAC2_NAME);
    }

    m_gadgetLoader.UnloadKernelModule(LIBCOMPOSITE_MODULE_NAME);

    if (iap2IsKernel314() || iap2IsKernel4xx()) {
        m_gadgetLoader.UnloadKernelModule(CONFIGFS_MODULE_NAME);
    }

    rc = iap2SwitchToDeviceMode(&m_RoleSwitchInfo, &m_UdcParam);
    mspin_log_printLn(eMspinVerbosityInfo,
         "%s() Switch USB OTG to Host  = %d", __FUNCTION__, rc);

    if (NULL != m_RoleSwitchInfo.vbusPower) {
        free(m_RoleSwitchInfo.vbusPower);
    }

    /* clear info */
    memset(&m_RoleSwitchInfo, 0, sizeof(m_RoleSwitchInfo));

    return rc;
}

int Iap2Connection::configureAccessoryBasics(iAP2AccessoryConfig_t* pConfig, bool boardTypeSD)
{
    U8 num_cfg = 0;
    S32 rc = IAP2_OK;
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(%p, boardType=%s) called", __FUNCTION__, pConfig, boardTypeSD ? "sd" : "ai");

    pConfig->iAP2iOSintheCar        = FALSE;
    pConfig->iAP2TransportType      = iAP2USBHOSTMODE; //to support EA native transport mode
    pConfig->iAP2AuthenticationType = iAP2AUTHI2C;
    IAP2TEST_Cfg *dcInfo = (IAP2TEST_Cfg *)m_iap2ConfigPfcfg.iAP2TestGetDevconfParameter(&num_cfg);
    if (dcInfo != NULL)
    {
        if((rc == IAP2_OK) && (dcInfo[IAP2_DC_AUTH_DEV_NAME].para.p_val != NULL))
        {
            rc = m_iap2Utilites.iap2CopyString(dcInfo[IAP2_DC_AUTH_DEV_NAME].para.p_val,&pConfig->iAP2AuthDevicename);
        }
        if((rc == IAP2_OK) && (dcInfo[IAP2_DC_AUTH_IOCTL_REG].para.p_val != NULL))
        {
            rc = m_iap2Utilites.iap2CopyString(dcInfo[IAP2_DC_AUTH_IOCTL_REG].para.p_val, &pConfig->iAP2AuthIoctlRegAddr);
        }
        if((rc == IAP2_OK) && (dcInfo[IAP2_DC_AUTH_RESET].para.p_val != NULL))
        {
            rc = m_iap2Utilites.iap2CopyString(dcInfo[IAP2_DC_AUTH_RESET].para.p_val,&pConfig->iAP2AuthGPIOReset);
        }
        if((rc == IAP2_OK) && (dcInfo[IAP2_DC_AUTH_READY].para.p_val != NULL))
        {
            rc = m_iap2Utilites.iap2CopyString(dcInfo[IAP2_DC_AUTH_READY].para.p_val, &pConfig->iAP2AuthGPIOReady);
        }
        if(rc == IAP2_OK)
        {
            pConfig->iAP2AuthShortWait    = dcInfo[IAP2_DC_AUTH_SHORT_WAIT].para.val;
            pConfig->iAP2AuthWait         = dcInfo[IAP2_DC_AUTH_WAIT].para.val;
            pConfig->iAP2AuthLongWait     = dcInfo[IAP2_DC_AUTH_LONG_WAIT].para.val;
        }
        m_iap2ConfigPfcfg.iAP2TestFreeDevconfParameter(dcInfo,num_cfg);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s():Failed to set Authentication Details of Accessory", __FUNCTION__);

        rc = IAP2_ERR_NO_MEM;
    }

    //Set accessory power configuration
    pConfig->iAP2AvailableCurrentForDevice                 = 2100;
    pConfig->iAP2DeviceBatteryShouldChargeIfPowerIsPresent = TRUE;
    pConfig->iAP2MaximumcurrentDrawnFromAccessory          = TRUE;
    pConfig->iAP2DeviceBatteryWillChargeIfPowerIsPresent   = TRUE;
    pConfig->iAP2AccessoryPowerMode                        = TRUE;
    pConfig->iAP2FileXferRcvAsStream                       = FALSE;
    pConfig->iAP2EANativeTransport                         = TRUE;

    if (TRUE == boardTypeSD)
    {
        pConfig->iAP2UsbOtgGPIOPower = (U8*)(VP)MYSPIN_DEMO_IAP2_ACC_CONFG_TRANS_USB_OTG_GPIO_POWER_SD;
    }
    else
    {
        pConfig->iAP2UsbOtgGPIOPower = (U8*)(VP)MYSPIN_DEMO_IAP2_ACC_CONFG_TRANS_USB_OTG_GPIO_POWER_AI;
    }

    pConfig->iAP2EAPSupported                              = FALSE;
    pConfig->iAP2FileXferSupported                         = FALSE;
    pConfig->useConfigFS                                   = TRUE;

    return rc;
}

S32 Iap2Connection::configureInfo(iAP2AccessoryInfo_t* pAccInfo, iAP2TransportType_t transportType)
{
    unsigned int i;
    S32 result = IAP2_OK;
    U8 AccessoryName[] = {MYSPIN_DEMO_IAP2_ACC_INFO_NAME};
    U8 ModelIdentifier[] = {MYSPIN_DEMO_IAP2_ACC_INFO_MODEL_IDENTIFIER};
    U8 Manufacturer[] = {MYSPIN_DEMO_IAP2_ACC_INFO_MANUFACTURER};
    U8 SerialNumber[] = {MYSPIN_DEMO_IAP2_ACC_INFO_SERIAL_NUM};
    U8 FirmwareVersion[] = {MYSPIN_DEMO_IAP2_ACC_INFO_FW_VER};
    U8 HardwareVersion[] = {MYSPIN_DEMO_IAP2_ACC_INFO_HW_VER};
    U8 idVendor[] = {MYSPIN_DEMO_IAP2_ACC_INFO_VENDOR_ID};
    U8 idProduct[] = {MYSPIN_DEMO_IAP2_ACC_INFO_PRODUCT_ID};
    U8 bcdDevice[] = {MYSPIN_DEMO_IAP2_ACC_INFO_BCD_DEVICE};
    U8 initEndpoint[] = {MYSPIN_DEMO_IAP2_ACC_INFO_INIT_ENDPOINT};
    U8 CurrentLanguage[] = {"de"};
    U16 SupportedLanguageCnt = 3;
    U8 SupportedLanguage[][3] = {"en", "de", "fr"};
    U8 iOSAppName[] = {MSPIN_DEMO_IAP2_APP_NAME};
    U8 iOSAppIdentifier = 1;
    U32 SupportediOSAppCount = 1;

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(%p, type=%d)", __FUNCTION__, pAccInfo, transportType);

    memset(pAccInfo, 0, sizeof(iAP2AccessoryInfo_t));

    //Accessory name
    result = m_iap2Utilites.iap2CopyString(AccessoryName, &(pAccInfo->iAP2AccessoryName));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Model identifier
    result = m_iap2Utilites.iap2CopyString(ModelIdentifier, &(pAccInfo->iAP2AccessoryModelIdentifier));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Manufacturer
    result = m_iap2Utilites.iap2CopyString(Manufacturer, &(pAccInfo->iAP2AccessoryManufacturer));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Serial number
    result = m_iap2Utilites.iap2CopyString(SerialNumber, &(pAccInfo->iAP2AccessorySerialNumber));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Firmware version
    result = m_iap2Utilites.iap2CopyString(FirmwareVersion, &(pAccInfo->iAP2AccessoryFirmwareVersion));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Hardware version
    result = m_iap2Utilites.iap2CopyString(HardwareVersion, &(pAccInfo->iAP2AccessoryHardwareVersion));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Vendor ID
    result = m_iap2Utilites.iap2CopyString(idVendor, &(pAccInfo->iAP2AccessoryVendorId));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Product ID
    result = m_iap2Utilites.iap2CopyString(idProduct, &(pAccInfo->iAP2AccessoryProductId));
    if (IAP2_OK != result)
    {
        return result;
    }

    //BCD device
    result = m_iap2Utilites.iap2CopyString(bcdDevice, &(pAccInfo->iAP2AccessoryBcdDevice));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Init end point
    result = m_iap2Utilites.iap2CopyString(initEndpoint, &(pAccInfo->iAP2InitEndPoint));
    if (IAP2_OK != result)
    {
        return result;
    }

    //Messages received/sent
    if (transportType == iAP2USBHOSTMODE)
    {
        //Messages sent by accessory
        pAccInfo->iAP2CommandsUsedByApplication = (U16*) calloc(1, sizeof(mspin_demo_USBHostModeMsgSentByAcc));
        if(pAccInfo->iAP2CommandsUsedByApplication != NULL)
        {
            memcpy(pAccInfo->iAP2CommandsUsedByApplication, mspin_demo_USBHostModeMsgSentByAcc,
                    sizeof(mspin_demo_USBHostModeMsgSentByAcc));
            pAccInfo->iAP2CommandsUsedByApplication_length = sizeof(mspin_demo_USBHostModeMsgSentByAcc);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p) FATAL ERROR: Failed to allocate memory for messages sent by accessory",
                    __FUNCTION__, pAccInfo);
            return IAP2_ERR_NO_MEM;
        }

        //Messages received from device
        pAccInfo->iAP2CallbacksExpectedFromDevice = (U16*) calloc(1, sizeof(mspin_demo_USBHostModeMsgRecvFromDevice) );
        if (pAccInfo->iAP2CallbacksExpectedFromDevice != NULL)
        {
            memcpy(pAccInfo->iAP2CallbacksExpectedFromDevice, mspin_demo_USBHostModeMsgRecvFromDevice, sizeof(mspin_demo_USBHostModeMsgRecvFromDevice) );
            pAccInfo->iAP2CallbacksExpectedFromDevice_length = sizeof(mspin_demo_USBHostModeMsgRecvFromDevice);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p) FATAL ERROR: Failed to allocate memory for messages received by accessory",
                    __FUNCTION__, pAccInfo);
            return IAP2_ERR_NO_MEM;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p) FATAL ERROR: Wrong configuration: Transport type must be USB Host Mode!",
                __FUNCTION__, pAccInfo);
        return IAP2_BAD_PARAMETER;
    }

    //Current language
    result = m_iap2Utilites.iap2CopyString(CurrentLanguage, &(pAccInfo->iAP2CurrentLanguage));
    if (IAP2_OK != result)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p) FATAL ERROR: Failed to allocate memory for current language",
                __FUNCTION__, pAccInfo);
        return result;
    }

    //Supported languages
    pAccInfo->iAP2SupportedLanguageCount = SupportedLanguageCnt;

    pAccInfo->iAP2SupportedLanguage = (U8**) calloc(pAccInfo->iAP2SupportedLanguageCount, sizeof(U8*));
    if (pAccInfo->iAP2SupportedLanguage != NULL)
    {
        for (i=0; i < pAccInfo->iAP2SupportedLanguageCount; i++)
        {
            result = m_iap2Utilites.iap2CopyString(&SupportedLanguage[i][0], &(pAccInfo->iAP2SupportedLanguage[i]));
            if (IAP2_OK != result)
            {
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(%p) FATAL ERROR: Failed to allocate memory for supported language %d",
                        __FUNCTION__, pAccInfo, i);
                return result;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p) FATAL ERROR: Failed to allocate memory for supported languages",
                __FUNCTION__, pAccInfo);
        return IAP2_ERR_NO_MEM;
    }

    pAccInfo->iAP2MaximumCurrentDrawnFromDevice = 0;

    //iOS in the Car is disabled
    pAccInfo->iAP2SupportsiOSintheCar = FALSE;

    //Supported iOS applications
    pAccInfo->iAP2SupportediOSAppCount = SupportediOSAppCount;
    pAccInfo->iAP2iOSAppInfo = (iAP2iOSAppInfo_t*) calloc(pAccInfo->iAP2SupportediOSAppCount, sizeof(iAP2iOSAppInfo_t));
    if (pAccInfo->iAP2iOSAppInfo != NULL)
    {
        for (i=0; i < pAccInfo->iAP2SupportediOSAppCount; i++)
        {
            pAccInfo->iAP2iOSAppInfo[i].iAP2EANativeTransport = TRUE; //must be true otherwise identification gets rejected
            pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppIdentifier = iOSAppIdentifier;
            pAccInfo->iAP2iOSAppInfo[i].iAP2EAPMatchAction = IAP2_USER_APP_MATCH; //No prompt but user can manually search for application in Settings -> General -> About
            result = m_iap2Utilites.iap2CopyString(iOSAppName, &(pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName));
            if (IAP2_OK != result)
            {
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(%p) FATAL ERROR: Failed to allocate memory for iOS app name",
                        __FUNCTION__, pAccInfo);
                return result;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p) FATAL ERROR: Failed to allocate memory for iOS app info",
                __FUNCTION__, pAccInfo);
        return IAP2_ERR_NO_MEM;
    }

//    //App bundle seed identifier
//    result = iap2CopyString(iOSAppBundleSeedId, &(pAccInfo->iAP2PreferredAppBundleSeedIdentifier));
//    if (IAP2_OK != result)
//    {
//        return result;
//    }

    return IAP2_OK;
}

void Iap2Connection::configureCallbacks(iAP2SessionCallbacks_t* pCallbacks)
{
    mspin_log_printLn(eMspinVerbosityVerbose, "%s(%p)", __FUNCTION__, pCallbacks);

    pCallbacks->iAP2AuthenticationFailed_cb                   = &authenticationFailed_CB;
    pCallbacks->iAP2AuthenticationSucceeded_cb                = &authenticationSucceeded_CB;
    pCallbacks->iAP2RequestAuthenticationCertificate_cb       = NULL;
    pCallbacks->iAP2RequestAuthenticationChallengeResponse_cb = NULL;
    pCallbacks->iAP2StartIdentification_cb                    = NULL;
    pCallbacks->iAP2IdentificationAccepted_cb                 = &identificationAccepted_CB;
    pCallbacks->iAP2IdentificationRejected_cb                 = &identificationRejected_CB;
    pCallbacks->iAP2AssistiveTouchInformation_cb              = NULL;
    pCallbacks->iAP2BluetoothConnectionUpdate_cb              = NULL;
    pCallbacks->iAP2DeviceAuthenticationCertificate_cb        = NULL;
    pCallbacks->iAP2DeviceAuthenticationResponse_cb           = NULL;
    pCallbacks->iAP2DeviceInformationUpdate_cb                = NULL;
    pCallbacks->iAP2DeviceLanguageUpdate_cb                   = NULL;
    pCallbacks->iAP2StartExternalAccessoryProtocolSession_cb  = NULL; //EAP start session callbacks will not be needed
    pCallbacks->iAP2StopExternalAccessoryProtocolSession_cb   = NULL; //EAP stop session callbacks will not be needed
    pCallbacks->iAP2DeviceHIDReport_cb                        = NULL;
    pCallbacks->iAP2StartLocationInformation_cb               = NULL;
    pCallbacks->iAP2StopLocationInformation_cb                = NULL;
    pCallbacks->iAP2MediaLibraryInformation_cb                = NULL;
    pCallbacks->iAP2MediaLibraryUpdate_cb                     = NULL;
    pCallbacks->iAP2NowPlayingUpdateParameter_cb              = NULL;
    pCallbacks->iAP2PowerUpdate_cb                            = &powerUpdate_CB;
    pCallbacks->iAP2TelephonyCallStateInformation_cb          = NULL;
    pCallbacks->iAP2TelephonyUpdate_cb                        = NULL;
    pCallbacks->iAP2USBDeviceModeAudioInformation_cb          = NULL;
    pCallbacks->iAP2StartVehicleStatusUpdates_cb              = NULL;
    pCallbacks->iAP2StopVehicleStatusUpdates_cb               = NULL;
    pCallbacks->iAP2VoiceOverCursorUpdate_cb                  = NULL;
    pCallbacks->iAP2VoiceOverUpdate_cb                        = NULL;
    pCallbacks->iAP2WiFiInformation_cb                        = NULL;
}

void Iap2Connection::configureStackCallbacks(iAP2StackCallbacks_t* pStackCallbacks)
{
    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(%p)", __FUNCTION__, pStackCallbacks);
    pStackCallbacks->p_iAP2DeviceState_cb = &updateDeviceState_CB;
}

void Iap2Connection::configureEANativeTransportCallbacks(iAP2EANativeTransportCallbacks_t* pEANativeTransportCallbacks)
{
    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(%p)", __FUNCTION__, pEANativeTransportCallbacks);

    pEANativeTransportCallbacks->p_iAP2StartEANativeTransport_cb = &startEANativeTransport_CB;
    pEANativeTransportCallbacks->p_iAP2StopEANativeTransport_cb = &stopEANativeTransport_CB;
}

int Iap2Connection::configureAccessory(iAP2InitParam_t* pInitParam, bool boardTypeSD, void* pContext)
{
    S32 rc = IAP2_OK;
    if (pInitParam)
    {
        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(%p, %s board, ctx=%p)",
                __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);

        pInitParam->iAP2ContextCallback = pContext;

        //Basic configuration
        pInitParam->p_iAP2AccessoryConfig = (iAP2AccessoryConfig_t*)calloc(1, sizeof(iAP2AccessoryConfig_t));
        if (NULL == pInitParam->p_iAP2AccessoryConfig)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p, %s board, ctx=%p) FATAL ERROR: Failed to allocate buffer for p_iAP2AccessoryConfig",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
            return IAP2_ERR_NO_MEM;
        }

        rc = configureAccessoryBasics(pInitParam->p_iAP2AccessoryConfig, boardTypeSD);
        if(rc != IAP2_OK)
        {
            return IAP2_BAD_PARAMETER;
        }
        //Configure accessory info
        pInitParam->p_iAP2AccessoryInfo = (iAP2AccessoryInfo_t*)calloc(1, sizeof(iAP2AccessoryInfo_t));
        if (NULL == pInitParam->p_iAP2AccessoryInfo)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p, %s board, ctx=%p) FATAL ERROR: Failed to allocate buffer for p_iAP2AccessoryInfo",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
            return IAP2_ERR_NO_MEM;
        }

        configureInfo(pInitParam->p_iAP2AccessoryInfo,
                pInitParam->p_iAP2AccessoryConfig->iAP2TransportType);

        //Configure iAP2 callbacks
        pInitParam->p_iAP2CSCallbacks = (iAP2SessionCallbacks_t*)calloc(1, sizeof(iAP2SessionCallbacks_t));
        if (NULL == pInitParam->p_iAP2CSCallbacks)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p, %s board, ctx=%p) FATAL ERROR: Failed to allocate buffer for p_iAP2CSCallbacks",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
            return IAP2_ERR_NO_MEM;
        }
        configureCallbacks(pInitParam->p_iAP2CSCallbacks);

        //Configure transfer callbacks is not required because iAP2FileXferSupported should be set to false)
        if (pInitParam->p_iAP2AccessoryConfig->iAP2FileXferSupported)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(%p, %s board, ctx=%p) ERROR: iAP2FileXferSupported should be disabled but is enabled!",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
        }

        //Configure stack callbacks
        pInitParam->p_iAP2StackCallbacks = (iAP2StackCallbacks_t*)calloc(1, sizeof(iAP2StackCallbacks_t));
        if (NULL == pInitParam->p_iAP2StackCallbacks)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p, %s board, ctx=%p) FATAL ERROR: Failed to allocate memory for iAP2StackCallbacks",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
            return IAP2_ERR_NO_MEM;
        }
        configureStackCallbacks(pInitParam->p_iAP2StackCallbacks);

        //Configure EA native transport callbacks (do not use EAP callbacks)
        pInitParam->p_iAP2EANativeTransportCallbacks = (iAP2EANativeTransportCallbacks_t*)calloc(1, sizeof(iAP2EANativeTransportCallbacks_t));
        if (NULL == pInitParam->p_iAP2EANativeTransportCallbacks)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(%p, %s board, ctx=%p) FATAL ERROR: Failed to allocate memory for iAP2EANativeTransportCallbacks",
                    __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
            return IAP2_ERR_NO_MEM ;
        }
        configureEANativeTransportCallbacks(pInitParam->p_iAP2EANativeTransportCallbacks);

        return IAP2_OK;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p, %s board, ctx=%p) FATAL ERROR: Accessory is NULL",
                __FUNCTION__, pInitParam, boardTypeSD ? "SD" : "AI", pContext);
        return IAP2_BAD_PARAMETER;
    }
}

pthread_t Iap2Connection::createThread(void* threadFunction, const char* thread_name, void* exinf)
{
    S32 rc = -1;
    pthread_t thread = 0;
    pthread_attr_t attr;
    (void)thread_name; //unused

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to init thread attributes", __FUNCTION__);
        return 0;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return 0;
    }

    rc = pthread_create(&thread, &attr, (void *(*)(void *))threadFunction, exinf);
    //usleep(100); //needed?

    (void)pthread_attr_destroy(&attr);

    return thread;
}

void Iap2Connection::monitorDevices_CB(void* exinf)
{
    iAP2Device_t* iap2device = (iAP2Device_t*)exinf;
    S32 rc = IAP2_OK;
    U32 retry_count = 0;
    (void)exinf;
    (void)rc;

    prctl(PR_SET_NAME,"iap2Monitor",0,0,0);

    //Wait till device gets ready
    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(%p) thread started -> waiting for device", __FUNCTION__, exinf);

    while ((m_iAP2Device.testDeviceState != iAP2DeviceReady)
            && (m_iAP2Device.testDeviceState != iAP2LinkiAP1DeviceDetected)
            && (retry_count < 300)
            && !m_EndMonitoring)
    {
        Iap2Utilites::iap2Sleep(50);
        retry_count++;
    }

    if (m_EndMonitoring)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(%p) monitoring terminated at retry_count=%d",
                __FUNCTION__, exinf, retry_count);
    }

    if (m_iAP2Device.testDeviceState == iAP2DeviceReady)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(%p) device attached (retry=%d)",
                __FUNCTION__, exinf, retry_count);
        rc = IAP2_OK;
    }
    else if (m_iAP2Device.testDeviceState == iAP2LinkiAP1DeviceDetected)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%p) ERROR: iAP1 device detected (retry=%d)-> Send BAD DETECT ACK to device because iAP1 isn't supported",
                __FUNCTION__, exinf, retry_count);

        rc = iAP2CanceliAP1Support(iap2device);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(%p) device not attached [state: %d | retry: %d]",
                __FUNCTION__, exinf, m_iAP2Device.testDeviceState, retry_count);
    }

    //Cleanup
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(%p) is terminating", __FUNCTION__, exinf);

    pthread_exit((void*)exinf);
}


void Iap2Connection::registerDLT()
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() register iAP2 DLT contexts", __FUNCTION__);

    IAP2REGISTERCTXTWITHDLT(); //mySPIN TA currently always registers a DLT context so we don't need any here
}

void Iap2Connection::unregisterDLT()
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() unregister iAP2 DLT contexts", __FUNCTION__);

    IAP2DEREGISTERCTXTWITHDLT();
}

S32 Iap2Connection::connect(std::string serial,
        iOSDeviceReady deviceReadyCB,
        iOSDeviceDisconnected deviceDisconnectedCB,
        EAPSessionStarted eapSessionStartedCB,
        EAPSessionStopped eapSessionStoppedCB,
        mspin_demo_connectionParameter_t *pDevice)
{
    iAP2Device_t* p_iAP2Device = NULL;
    S32 rc = IAP2_OK;

    if (!pDevice)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: device is NULL",
                __FUNCTION__, serial.c_str(), pDevice);
        return IAP2_BAD_PARAMETER;
    }

    m_iap2Utilites.iap2StopPolling(FALSE); //make sure polling is running

    m_Connect_CB = deviceReadyCB;
    m_Disconnected_CB = deviceDisconnectedCB;
    m_EAPSessionStarted_CB = eapSessionStartedCB;
    m_EAPSessionStopped_CB = eapSessionStoppedCB;

    memset(&m_iAP2Device, 0, sizeof(m_iAP2Device));

    //Initialize accessory
    memset(&m_iAP2InitParameter, 0, sizeof(iAP2InitParam_t));

    rc = configureAccessory(&m_iAP2InitParameter, pDevice->boardTypeSD, (void*)pDevice);
    if (IAP2_OK != rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: Failed to configure accessory with rc=%d",
                __FUNCTION__, serial.c_str(), pDevice, rc);
        return rc;
    }

    //Set serial
    strncpy((char*)m_iAP2InitParameter.iAP2DeviceId, serial.c_str(), serial.length());

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(s/n='%s', device=%p) device id set to '%s'",
            __FUNCTION__, serial.c_str(), pDevice, m_iAP2InitParameter.iAP2DeviceId);

    //Initialize iAP2 device structure
    p_iAP2Device = iAP2InitDeviceStructure(&m_iAP2InitParameter);

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(s/n='%s', device=%p) device=%p created => initialize device connection",
            __FUNCTION__, serial.c_str(), pDevice, p_iAP2Device);

    if (NULL != p_iAP2Device)
    {
        m_iAP2Device.p_iAP2Device = p_iAP2Device; //store the device for later calls

        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(s/n='%s', device=%p) initialized device structure is %p",
                __FUNCTION__, serial.c_str(), pDevice, p_iAP2Device);

        //Switch headunit to gadget mode
        rc = switchOTGPortToDevice();
        if (IAP2_OK != rc)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(s/n='%s', device=%p) ERROR: Failed to switch to host mode %d",
                    __FUNCTION__, serial.c_str(), pDevice, rc);
        }

        rc = iAP2InitDeviceConnection(p_iAP2Device);

        if (IAP2_OK == rc)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(s/n='%s', device=%p) Device connection initialized",
                    __FUNCTION__, serial.c_str(), pDevice);
        }
        else
        {
            m_iAP2Device.p_iAP2Device = NULL;
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(s/n='%s', device=%p) ERROR: Failed to initialize device connection with rc=%d",
                    __FUNCTION__, serial.c_str(), pDevice, rc);
            return rc;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(s/n='%s', device=%p) ERROR: Failed to create iAP2 device",
                __FUNCTION__, serial.c_str(), pDevice);
        return IAP2_ERR_NO_MEM;
    }

    //Create message queues
    rc = m_iap2Utilites.createMsgQueue(&m_MyspinMQFD, MYSPIN_MQ_NAME, O_CREAT | O_RDWR);
    if (rc != IAP2_OK)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: Failed to create mq %s with %d",
                __FUNCTION__, serial.c_str(), pDevice, MYSPIN_MQ_NAME, rc);
        return rc;
    }

    rc = m_iap2Utilites.createMsgQueue(&m_iAP2Device.mqAppTskFd, MYSPIN_MQ_NAME_APP_TSK, O_CREAT | O_RDWR);
    if (rc != IAP2_OK)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: Failed to create mq %s with %d",
                __FUNCTION__, serial.c_str(), pDevice, MYSPIN_MQ_NAME_APP_TSK, rc);
        return rc;
    }

    //Start polling thread
    m_PollThreadID = createThread((void*)Iap2Utilites::iap2Poll, "iap2Poll",
            m_iAP2Device.p_iAP2Device);
    if (0 == m_PollThreadID)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: Failed to create polling thread",
                __FUNCTION__, serial.c_str(), pDevice);
        return IAP2_CTL_ERROR;
    }

    //Start application thread
    m_EndMonitoring = FALSE;
    m_AppThreadID = createThread((void*)monitorDevices_CB, "iap2Monitor", p_iAP2Device);
    if (0 == m_AppThreadID)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(s/n='%s', device=%p) FATAL ERROR: Failed to create application thread",
                __FUNCTION__, serial.c_str(), pDevice);
        return IAP2_CTL_ERROR;
    }

    return rc;
}

void Iap2Connection::disconnect(void)
{
    S32 rc = -1;
    void* status;

    mspin_log_printLn(eMspinVerbosityInfo, "%s() called", __FUNCTION__);

    Iap2Utilites::iap2StopPolling(TRUE);

    //Send stop poll
    (void)m_iap2Utilites.iap2Write(m_MyspinMQFD, MYSPIN_MQ_CMD_STOP_POLL, sizeof(MYSPIN_MQ_CMD_STOP_POLL));

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() stop poll sent => join poll thread=%d", __FUNCTION__, m_PollThreadID);

    // and join poll thread
    if (m_PollThreadID > 0)
    {
       (void)pthread_join(m_PollThreadID, &status);
       m_PollThreadID = 0;

       mspin_log_printLn(eMspinVerbosityDebug,
               "%s() poll thread joined", __FUNCTION__);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s() WARNING: Poll thread id=%d is invalid",
                __FUNCTION__, m_PollThreadID);
    }

    m_EndMonitoring = TRUE; //stop iAP2 device state monitoring

    //Clean message queues
    if (m_MyspinMQFD > 0)
    {
        rc = mq_close(m_MyspinMQFD);
        rc = mq_unlink(MYSPIN_MQ_NAME);
        m_MyspinMQFD = -1;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() mySPIN mq joined", __FUNCTION__);

    if (m_iAP2Device.mqAppTskFd > 0)
    {
        rc = mq_close(m_iAP2Device.mqAppTskFd);
        rc = mq_unlink(MYSPIN_MQ_NAME_APP_TSK);
        m_iAP2Device.mqAppTskFd = -1;
    }

    //Do cleanup in iAP2
    if (m_iAP2Device.p_iAP2Device)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s() mq app tsk joined -> next: disconnect iAP2 device", __FUNCTION__);

        //Disconnect device
        rc = iAP2DisconnectDevice(m_iAP2Device.p_iAP2Device);
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s() iAP2DisconnectDevice returned with rc=%d", __FUNCTION__, rc);

        //Switch headunit back to host mode
        switchOTGPortToHost();

        //De-initialize the device structure
        rc = iAP2DeInitDeviceStructure(m_iAP2Device.p_iAP2Device);

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s() iAP2DeInitDeviceStructure returned with rc=%d", __FUNCTION__, rc);

        m_iAP2Device.p_iAP2Device = NULL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: iap2Device is NULL", __FUNCTION__);
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() join device monitoring thread", __FUNCTION__);

    //Join device monitoring thread
    if (m_AppThreadID > 0)
    {
        (void)pthread_join(m_AppThreadID, &status);
        m_AppThreadID = 0;
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s() device monitoring thread joined", __FUNCTION__);
    }

    m_iap2Utilites.resetInitialParameter(&m_iAP2InitParameter);

    //Reset values
    m_EAPSessionStarted_CB = NULL;
    m_EAPSessionStopped_CB = NULL;
    m_Disconnected_CB = NULL;
    m_Connect_CB = NULL;
}

S32 Iap2Connection::requestAppLaunch(bool withAlert)
{
    S32 rc = IAP2_CTL_ERROR;
    U8* iOSBundleID = NULL;
    U8* iOSAppNameContainer[1] = { NULL };
    iAP2AppLaunchMethod appLaunchMethod = IAP2_LAUNCH_WITH_USER_ALERT;

    if (!withAlert)
    {
        appLaunchMethod = IAP2_LAUNCH_WITHOUT_USER_ALERT;
    }

    iAP2RequestAppLaunchParameter appLaunchParameter;

    memset(&appLaunchParameter, 0, sizeof(iAP2RequestAppLaunchParameter));
    iOSBundleID = (U8*) calloc(strlen(MSPIN_DEMO_IAP2_BUNDLE_ID) + 1, sizeof(U8)); //length+1 for '\0'

    if (!iOSBundleID)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to allocate memory for iOS app name",
                __FUNCTION__);
        return IAP2_ERR_NO_MEM;
    }

    strcpy((char*)iOSBundleID, MSPIN_DEMO_IAP2_BUNDLE_ID);
    iOSAppNameContainer[0] = iOSBundleID;

    if (m_iAP2Device.p_iAP2Device)
    {
        //Set app name (we have only one!)
        appLaunchParameter.iAP2AppBundleID_count = 1;
        appLaunchParameter.iAP2AppBundleID = iOSAppNameContainer;
        appLaunchParameter.iAP2LaunchAlert_count = 1;
        appLaunchParameter.iAP2LaunchAlert = &appLaunchMethod;

        rc = iAP2RequestAppLaunch(m_iAP2Device.p_iAP2Device, &appLaunchParameter);
        if (IAP2_OK != rc)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to request app launch with rc=%d",
                    __FUNCTION__);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s() app='%s' requested to launch",
                    __FUNCTION__, iOSBundleID);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: iap2Device is NULL",
                __FUNCTION__);
    }

    free(iOSBundleID);
    iOSBundleID = NULL;

    return rc;
}

S32 Iap2Connection::powerSourceUpdate(void)
{
    S32 rc = IAP2_CTL_ERROR;
    iAP2PowerSourceUpdateParameter theiAP2PowerSourceUpdateParameter;

    memset(&theiAP2PowerSourceUpdateParameter, 0, sizeof(iAP2PowerSourceUpdateParameter));

    if (m_iAP2Device.p_iAP2Device)
    {
        theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice = (U16*) calloc(1, sizeof(U16));
        if (!theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to allocate memory for iAP2AvailableCurrentForDevice",
                    __FUNCTION__);
            return IAP2_ERR_NO_MEM;
        }
        theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent = (U8*) calloc(1, sizeof(U8));
        if (!theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to allocate memory for iAP2DeviceBatteryShouldChargeIfPowerIsPresent",
                    __FUNCTION__);
            return IAP2_ERR_NO_MEM;
        }

        *(theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice) = 2100;
        theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice_count++;

        *(theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent) = 1;
        theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent_count++;

        rc = iAP2PowerSourceUpdate(m_iAP2Device.p_iAP2Device, &theiAP2PowerSourceUpdateParameter);
        if (IAP2_OK != rc)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to request app launch with rc=%d",
                    __FUNCTION__);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: iap2Device is NULL",
                __FUNCTION__);
    }

    free(theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice);
    theiAP2PowerSourceUpdateParameter.iAP2AvailableCurrentForDevice = NULL;
    free(theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent);
    theiAP2PowerSourceUpdateParameter.iAP2DeviceBatteryShouldChargeIfPowerIsPresent = NULL;

    return rc;
}


#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED
